<?php
/**
 * Sign Up Page
 * Firebase REST API authentication — no CDN/SDK.
 * Collects first name for password recovery verification.
 */

session_start();

require_once __DIR__ . '/config/firebase_config.php';
require_once __DIR__ . '/backend/auth/firebase_auth.php';
require_once __DIR__ . '/backend/auth/user_store.php';
require_once __DIR__ . '/backend/auth/auth_ui.php';

// Redirect if Firebase is not configured
if (!FirebaseConfig::isConfigured()) {
    header('Location: setup.php');
    exit;
}

// Redirect if already signed in
if (!empty($_SESSION['firebase_user'])) {
    header('Location: dashboard.php');
    exit;
}

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $firstName       = trim($_POST['first_name'] ?? '');
    $email           = trim($_POST['email'] ?? '');
    $password         = $_POST['password'] ?? '';
    $confirmPassword  = $_POST['confirm_password'] ?? '';

    // Validation
    if (empty($firstName) || empty($email) || empty($password) || empty($confirmPassword)) {
        $error = 'All fields are required.';
    } elseif (strlen($firstName) < 2) {
        $error = 'First name must be at least 2 characters.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Please enter a valid email address.';
    } elseif (strlen($password) < 6) {
        $error = 'Password must be at least 6 characters.';
    } elseif ($password !== $confirmPassword) {
        $error = 'Passwords do not match.';
    } else {
        try {
            $auth = new FirebaseAuth();
            $result = $auth->signUp($email, $password, $firstName);

            if ($result['success']) {
                // Store user profile (email → first name) for password recovery
                UserStore::saveUser($email, $firstName);

                // Set session
                $_SESSION['firebase_user'] = [
                    'uid'          => $result['uid'],
                    'email'        => $result['email'],
                    'firstName'    => $result['firstName'],
                    'idToken'      => $result['idToken'],
                    'refreshToken' => $result['refreshToken'],
                    'loginTime'    => time(),
                ];

                header('Location: dashboard.php');
                exit;
            } else {
                $error = $result['error'];
            }
        } catch (Exception $e) {
            $error = $e->getMessage();
        }
    }
}

renderAuthHead('Create Account');
?>

<div class="auth-wrapper">
    <?php renderAuthBrand(); ?>

    <div class="auth-form-panel">
        <div class="auth-card">
            <div class="auth-header">
                <?php renderAuthLogo(); ?>
                <h1>Create Account</h1>
                <p>Start tracking your analytics today</p>
            </div>

            <div id="alertError" class="alert alert-error <?= $error ? 'show' : '' ?>">
                <span class="alert-icon">✕</span>
                <span id="errorText"><?= htmlspecialchars($error) ?></span>
            </div>

            <form method="POST" id="signupForm" novalidate>
                <div class="form-group">
                    <label class="form-label">First Name</label>
                    <input type="text" name="first_name" id="firstName" class="form-input"
                           placeholder="Enter your first name"
                           value="<?= htmlspecialchars($_POST['first_name'] ?? '') ?>"
                           required autocomplete="given-name" autofocus>
                    <div style="margin-top: 5px;">
                        <small style="color: var(--gray-400); font-size: 11px;">
                            📌 Remember this — it's needed for password recovery
                        </small>
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-label">Email Address</label>
                    <input type="email" name="email" id="email" class="form-input"
                           placeholder="you@example.com"
                           value="<?= htmlspecialchars($_POST['email'] ?? '') ?>"
                           required autocomplete="email">
                </div>

                <div class="form-group">
                    <label class="form-label">Password</label>
                    <div class="password-wrapper">
                        <input type="password" name="password" id="password" class="form-input"
                               placeholder="Minimum 6 characters"
                               required autocomplete="new-password" minlength="6">
                        <button type="button" class="password-toggle" onclick="togglePassword('password', this)" tabindex="-1">
                            <?php renderPasswordToggleSVG(); ?>
                        </button>
                    </div>
                    <div class="password-strength" id="strengthMeter" style="display: none;">
                        <div class="strength-bar">
                            <div class="strength-fill" id="strengthFill"></div>
                        </div>
                        <span class="strength-text" id="strengthText"></span>
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-label">Confirm Password</label>
                    <div class="password-wrapper">
                        <input type="password" name="confirm_password" id="confirmPassword" class="form-input"
                               placeholder="Re-enter your password"
                               required autocomplete="new-password">
                        <button type="button" class="password-toggle" onclick="togglePassword('confirmPassword', this)" tabindex="-1">
                            <?php renderPasswordToggleSVG(); ?>
                        </button>
                    </div>
                </div>

                <button type="submit" class="btn btn-primary" id="submitBtn">
                    <div class="spinner"></div>
                    <span class="btn-text">Create Account</span>
                </button>
            </form>

            <div class="auth-footer">
                <p>Already have an account? <a href="signin.php">Sign In</a></p>
            </div>
        </div>
    </div>
</div>

<script>
    // Password visibility toggle
    function togglePassword(inputId, btn) {
        const input = document.getElementById(inputId);
        const isPassword = input.type === 'password';
        input.type = isPassword ? 'text' : 'password';
        btn.querySelector('.eye-open').style.display = isPassword ? 'none' : 'block';
        btn.querySelector('.eye-closed').style.display = isPassword ? 'block' : 'none';
    }

    // Password strength meter
    document.getElementById('password').addEventListener('input', function() {
        const meter = document.getElementById('strengthMeter');
        const fill = document.getElementById('strengthFill');
        const text = document.getElementById('strengthText');
        const val = this.value;

        if (val.length === 0) {
            meter.style.display = 'none';
            return;
        }

        meter.style.display = 'block';
        let score = 0;

        if (val.length >= 6) score++;
        if (val.length >= 10) score++;
        if (/[A-Z]/.test(val) && /[a-z]/.test(val)) score++;
        if (/[0-9]/.test(val)) score++;
        if (/[^A-Za-z0-9]/.test(val)) score++;

        fill.className = 'strength-fill';
        if (score <= 1) { fill.classList.add('weak'); text.textContent = 'Weak'; text.style.color = 'var(--danger)'; }
        else if (score === 2) { fill.classList.add('fair'); text.textContent = 'Fair'; text.style.color = 'var(--warning)'; }
        else if (score === 3) { fill.classList.add('good'); text.textContent = 'Good'; text.style.color = '#22c55e'; }
        else { fill.classList.add('strong'); text.textContent = 'Strong'; text.style.color = 'var(--success)'; }
    });

    // Client-side validation
    document.getElementById('signupForm').addEventListener('submit', function(e) {
        const firstName = document.getElementById('firstName').value.trim();
        const email = document.getElementById('email').value.trim();
        const password = document.getElementById('password').value;
        const confirmPassword = document.getElementById('confirmPassword').value;
        const alertEl = document.getElementById('alertError');
        const errorText = document.getElementById('errorText');

        let err = '';

        if (!firstName || firstName.length < 2) err = 'First name must be at least 2 characters.';
        else if (!email || !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) err = 'Please enter a valid email address.';
        else if (password.length < 6) err = 'Password must be at least 6 characters.';
        else if (password !== confirmPassword) err = 'Passwords do not match.';

        if (err) {
            e.preventDefault();
            errorText.textContent = err;
            alertEl.classList.add('show');
            alertEl.scrollIntoView({ behavior: 'smooth', block: 'center' });
            return;
        }

        // Show loading
        const btn = document.getElementById('submitBtn');
        btn.classList.add('loading');
        btn.disabled = true;
    });
</script>

<?php renderAuthFooter(); ?>
