<?php
/**
 * Sign In Page
 * Firebase REST API authentication — no CDN/SDK.
 */

session_start();

require_once __DIR__ . '/config/firebase_config.php';
require_once __DIR__ . '/backend/auth/firebase_auth.php';
require_once __DIR__ . '/backend/auth/auth_ui.php';

// Redirect if Firebase is not configured
if (!FirebaseConfig::isConfigured()) {
    header('Location: setup.php');
    exit;
}

// Redirect if already signed in
if (!empty($_SESSION['firebase_user'])) {
    header('Location: dashboard.php');
    exit;
}

$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email    = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';

    if (empty($email) || empty($password)) {
        $error = 'Please enter both email and password.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Please enter a valid email address.';
    } else {
        try {
            $auth = new FirebaseAuth();
            $result = $auth->signIn($email, $password);

            if ($result['success']) {
                $_SESSION['firebase_user'] = [
                    'uid'          => $result['uid'],
                    'email'        => $result['email'],
                    'firstName'    => $result['firstName'],
                    'idToken'      => $result['idToken'],
                    'refreshToken' => $result['refreshToken'],
                    'loginTime'    => time(),
                ];

                header('Location: dashboard.php');
                exit;
            } else {
                $error = $result['error'];
            }
        } catch (Exception $e) {
            $error = $e->getMessage();
        }
    }
}

renderAuthHead('Sign In');
?>

<div class="auth-wrapper">
    <?php renderAuthBrand(); ?>

    <div class="auth-form-panel">
        <div class="auth-card">
            <div class="auth-header">
                <?php renderAuthLogo(); ?>
                <h1>Welcome Back</h1>
                <p>Sign in to your analytics dashboard</p>
            </div>

            <div id="alertError" class="alert alert-error <?= $error ? 'show' : '' ?>">
                <span class="alert-icon">✕</span>
                <span id="errorText"><?= htmlspecialchars($error) ?></span>
            </div>

            <form method="POST" id="signinForm" novalidate>
                <div class="form-group">
                    <label class="form-label">Email Address</label>
                    <input type="email" name="email" id="email" class="form-input"
                           placeholder="you@example.com"
                           value="<?= htmlspecialchars($_POST['email'] ?? '') ?>"
                           required autocomplete="email" autofocus>
                </div>

                <div class="form-group">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 7px;">
                        <label class="form-label" style="margin-bottom: 0;">Password</label>
                        <a href="forgot_password.php" class="auth-link" style="font-size: 12px;">Forgot password?</a>
                    </div>
                    <div class="password-wrapper">
                        <input type="password" name="password" id="password" class="form-input"
                               placeholder="Enter your password"
                               required autocomplete="current-password">
                        <button type="button" class="password-toggle" onclick="togglePassword('password', this)" tabindex="-1">
                            <?php renderPasswordToggleSVG(); ?>
                        </button>
                    </div>
                </div>

                <div class="form-check">
                    <input type="checkbox" id="rememberMe" name="remember_me">
                    <label for="rememberMe">Keep me signed in for 30 days</label>
                </div>

                <button type="submit" class="btn btn-primary" id="submitBtn">
                    <div class="spinner"></div>
                    <span class="btn-text">Sign In</span>
                </button>
            </form>

            <div class="auth-footer">
                <p>Don't have an account? <a href="signup.php">Create Account</a></p>
                <p style="margin-top: 10px;">
                    <a href="setup.php" style="font-weight: 400; color: var(--gray-400); font-size: 12px;">Firebase Setup</a>
                </p>
            </div>
        </div>
    </div>
</div>

<script>
    function togglePassword(inputId, btn) {
        const input = document.getElementById(inputId);
        const isPassword = input.type === 'password';
        input.type = isPassword ? 'text' : 'password';
        btn.querySelector('.eye-open').style.display = isPassword ? 'none' : 'block';
        btn.querySelector('.eye-closed').style.display = isPassword ? 'block' : 'none';
    }

    document.getElementById('signinForm').addEventListener('submit', function(e) {
        const email = document.getElementById('email').value.trim();
        const password = document.getElementById('password').value;
        const alertEl = document.getElementById('alertError');
        const errorText = document.getElementById('errorText');

        let err = '';
        if (!email || !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) err = 'Please enter a valid email address.';
        else if (!password) err = 'Please enter your password.';

        if (err) {
            e.preventDefault();
            errorText.textContent = err;
            alertEl.classList.add('show');
            return;
        }

        const btn = document.getElementById('submitBtn');
        btn.classList.add('loading');
        btn.disabled = true;
    });
</script>

<?php renderAuthFooter(); ?>
