<?php
/**
 * Firebase Setup Page
 * Allows admin to input Firebase credentials without storing them in source files.
 * Credentials are encrypted and stored securely on disk.
 */

session_start();

require_once __DIR__ . '/config/firebase_config.php';
require_once __DIR__ . '/backend/auth/auth_ui.php';

$message = '';
$messageType = '';
$isConfigured = FirebaseConfig::isConfigured();

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'save') {
        $apiKey     = trim($_POST['api_key'] ?? '');
        $authDomain = trim($_POST['auth_domain'] ?? '');
        $projectId  = trim($_POST['project_id'] ?? '');

        if (empty($apiKey) || empty($authDomain) || empty($projectId)) {
            $message = 'All fields are required.';
            $messageType = 'error';
        } else {
            $saved = FirebaseConfig::saveCredentials([
                'api_key'     => $apiKey,
                'auth_domain' => $authDomain,
                'project_id'  => $projectId,
            ]);

            if ($saved) {
                $message = 'Firebase credentials saved successfully! You can now use sign-up and sign-in.';
                $messageType = 'success';
                $isConfigured = true;
            } else {
                $message = 'Failed to save credentials. Check directory permissions.';
                $messageType = 'error';
            }
        }
    } elseif ($action === 'delete') {
        FirebaseConfig::deleteCredentials();
        $message = 'Credentials deleted. Firebase authentication is now disabled.';
        $messageType = 'success';
        $isConfigured = false;
    } elseif ($action === 'test') {
        // Test the Firebase API key by making a lightweight request
        $apiKey = FirebaseConfig::getApiKey();
        if ($apiKey) {
            $url = 'https://identitytoolkit.googleapis.com/v1/accounts:signUp?key=' . $apiKey;
            $ch = curl_init($url);
            curl_setopt_array($ch, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_POST           => true,
                CURLOPT_HTTPHEADER     => ['Content-Type: application/json'],
                CURLOPT_POSTFIELDS     => json_encode(['returnSecureToken' => false]),
                CURLOPT_TIMEOUT        => 10,
                CURLOPT_SSL_VERIFYPEER => true,
            ]);
            $body = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlError = curl_error($ch);
            curl_close($ch);

            if ($curlError) {
                $message = 'Connection failed: ' . $curlError;
                $messageType = 'error';
            } elseif ($httpCode === 400) {
                // 400 = API key is valid (Firebase returns 400 for missing email, which is expected)
                $response = json_decode($body, true);
                $errorMsg = $response['error']['message'] ?? '';
                if (strpos($errorMsg, 'API key not valid') !== false || strpos($errorMsg, 'API_KEY_INVALID') !== false) {
                    $message = 'API key is INVALID. Please check your Firebase API key.';
                    $messageType = 'error';
                } else {
                    $message = 'Connection successful! Firebase API key is valid and working.';
                    $messageType = 'success';
                }
            } elseif ($httpCode === 403) {
                $message = 'API key is invalid or Identity Toolkit API is not enabled.';
                $messageType = 'error';
            } else {
                $message = 'Unexpected response (HTTP ' . $httpCode . '). Please verify your credentials.';
                $messageType = 'error';
            }
        } else {
            $message = 'No credentials saved yet.';
            $messageType = 'error';
        }
    }
}

// Get existing credentials for display (masked)
$existingCreds = FirebaseConfig::getCredentials();

renderAuthHead('Firebase Setup');
?>

<div class="auth-wrapper">
    <?php renderAuthBrand(); ?>

    <div class="auth-form-panel">
        <div class="auth-card" style="max-width: 520px;">
            <div class="auth-header">
                <div class="auth-logo" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); box-shadow: 0 4px 12px rgba(245, 158, 11, 0.25);">
                    <svg viewBox="0 0 24 24" fill="white" xmlns="http://www.w3.org/2000/svg" width="28" height="28">
                        <path d="M12 1L3 5v6c0 5.55 3.84 10.74 9 12 5.16-1.26 9-6.45 9-12V5l-9-4zm0 2.18l7 3.12v4.7c0 4.83-3.13 9.37-7 10.5-3.87-1.13-7-5.67-7-10.5V6.3l7-3.12zM11 7v2h2V7h-2zm0 4v6h2v-6h-2z"/>
                    </svg>
                </div>
                <h1>Firebase Setup</h1>
                <p>Configure your Firebase project credentials</p>
            </div>

            <?php if ($message): ?>
                <div class="alert alert-<?= $messageType ?> show">
                    <span class="alert-icon">
                        <?= $messageType === 'success' ? '✓' : '✕' ?>
                    </span>
                    <?= htmlspecialchars($message) ?>
                </div>
            <?php endif; ?>

            <?php if ($isConfigured): ?>
                <!-- Status: Configured -->
                <div class="info-box" style="background: rgba(16, 185, 129, 0.08); border-left-color: var(--success);">
                    <strong>✓ Firebase is configured</strong><br>
                    Project: <strong><?= htmlspecialchars($existingCreds['project_id'] ?? '—') ?></strong><br>
                    Domain: <?= htmlspecialchars($existingCreds['auth_domain'] ?? '—') ?><br>
                    API Key: <?= htmlspecialchars(substr($existingCreds['api_key'] ?? '', 0, 10)) ?>•••••<br>
                    <small style="color: var(--gray-400);">Last updated: <?= htmlspecialchars($existingCreds['updated_at'] ?? '—') ?></small>
                </div>

                <!-- Test Connection -->
                <form method="POST" style="margin-bottom: 12px;">
                    <input type="hidden" name="action" value="test">
                    <button type="submit" class="btn btn-primary" style="background: linear-gradient(135deg, var(--success) 0%, #059669 100%); box-shadow: 0 4px 12px rgba(16, 185, 129, 0.25);">
                        <span class="btn-text">Test Connection</span>
                    </button>
                </form>

                <div class="auth-divider"><span>or update credentials</span></div>
            <?php else: ?>
                <div class="info-box">
                    <strong>How to get Firebase credentials:</strong><br>
                    1. Go to <a href="https://console.firebase.google.com" target="_blank" class="auth-link">Firebase Console</a><br>
                    2. Create or select a project<br>
                    3. Go to Project Settings → General<br>
                    4. Under "Your apps", click Web app (or add one)<br>
                    5. Copy the credentials below<br><br>
                    <strong>Important:</strong> Enable <em>Email/Password</em> under Authentication → Sign-in method
                </div>
            <?php endif; ?>

            <!-- Credentials Form -->
            <form method="POST" id="setupForm">
                <input type="hidden" name="action" value="save">

                <div class="form-group">
                    <label class="form-label">Firebase API Key *</label>
                    <input type="text" name="api_key" class="form-input"
                           placeholder="AIzaSyB..."
                           value="<?= htmlspecialchars($existingCreds['api_key'] ?? '') ?>"
                           required autocomplete="off" spellcheck="false">
                </div>

                <div class="form-group">
                    <label class="form-label">Auth Domain *</label>
                    <input type="text" name="auth_domain" class="form-input"
                           placeholder="your-project.firebaseapp.com"
                           value="<?= htmlspecialchars($existingCreds['auth_domain'] ?? '') ?>"
                           required autocomplete="off" spellcheck="false">
                </div>

                <div class="form-group">
                    <label class="form-label">Project ID *</label>
                    <input type="text" name="project_id" class="form-input"
                           placeholder="your-project-id"
                           value="<?= htmlspecialchars($existingCreds['project_id'] ?? '') ?>"
                           required autocomplete="off" spellcheck="false">
                </div>

                <button type="submit" class="btn btn-primary" style="margin-bottom: 12px;">
                    <span class="btn-text"><?= $isConfigured ? 'Update Credentials' : 'Save & Enable Authentication' ?></span>
                </button>
            </form>

            <?php if ($isConfigured): ?>
                <!-- Delete Credentials -->
                <form method="POST" onsubmit="return confirm('Are you sure? This will disable Firebase authentication.');">
                    <input type="hidden" name="action" value="delete">
                    <button type="submit" class="btn btn-outline" style="color: var(--danger); border-color: rgba(239,68,68,0.3);">
                        <span class="btn-text">Remove Credentials</span>
                    </button>
                </form>
            <?php endif; ?>

            <div class="auth-footer">
                <p>
                    <a href="signin.php">← Back to Sign In</a>
                </p>
            </div>
        </div>
    </div>
</div>

<?php renderAuthFooter(); ?>
