<?php
/**
 * Forgot Password Page
 * Verifies user identity via first name, then sends Firebase password reset email.
 */

session_start();

require_once __DIR__ . '/config/firebase_config.php';
require_once __DIR__ . '/backend/auth/firebase_auth.php';
require_once __DIR__ . '/backend/auth/user_store.php';
require_once __DIR__ . '/backend/auth/auth_ui.php';

// Redirect if Firebase is not configured
if (!FirebaseConfig::isConfigured()) {
    header('Location: setup.php');
    exit;
}

$error = '';
$success = '';
$step = 'verify'; // verify → reset sent

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email     = trim($_POST['email'] ?? '');
    $firstName = trim($_POST['first_name'] ?? '');

    if (empty($email) || empty($firstName)) {
        $error = 'Please enter both your email and first name.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Please enter a valid email address.';
    } else {
        // Verify the first name matches what was stored during signup
        if (!UserStore::verifyFirstName($email, $firstName)) {
            $error = 'The first name does not match our records for this email. Please check and try again.';
        } else {
            try {
                $auth = new FirebaseAuth();
                $result = $auth->sendPasswordReset($email);

                if ($result['success']) {
                    $success = 'Password reset email sent! Check your inbox (and spam folder) for a link to reset your password.';
                    $step = 'sent';
                } else {
                    $error = $result['error'];
                }
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
    }
}

renderAuthHead('Reset Password');
?>

<div class="auth-wrapper">
    <?php renderAuthBrand(); ?>

    <div class="auth-form-panel">
        <div class="auth-card">
            <div class="auth-header">
                <!-- Lock icon -->
                <div class="auth-logo" style="background: linear-gradient(135deg, #f59e0b 0%, #ea580c 100%); box-shadow: 0 4px 12px rgba(245, 158, 11, 0.25);">
                    <svg viewBox="0 0 24 24" fill="white" xmlns="http://www.w3.org/2000/svg" width="26" height="26">
                        <path d="M12 1C8.676 1 6 3.676 6 7v2H4v14h16V9h-2V7c0-3.324-2.676-6-6-6zm0 2c2.276 0 4 1.724 4 4v2H8V7c0-2.276 1.724-4 4-4zm0 10c1.1 0 2 .9 2 2s-.9 2-2 2-2-.9-2-2 .9-2 2-2z"/>
                    </svg>
                </div>
                <h1>Reset Password</h1>
                <p>Verify your identity to receive a reset link</p>
            </div>

            <?php if ($error): ?>
                <div class="alert alert-error show">
                    <span class="alert-icon">✕</span>
                    <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <?php if ($step === 'sent'): ?>
                <!-- Success state -->
                <div class="alert alert-success show">
                    <span class="alert-icon">✓</span>
                    <?= htmlspecialchars($success) ?>
                </div>

                <div style="text-align: center; padding: 20px 0;">
                    <div style="width: 72px; height: 72px; background: var(--success-bg); border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 20px; font-size: 32px;">
                        ✉️
                    </div>
                    <p style="color: var(--gray-600); font-size: 14px; line-height: 1.6; max-width: 320px; margin: 0 auto;">
                        We've sent a password reset link to <strong style="color: var(--gray-800);"><?= htmlspecialchars($_POST['email'] ?? '') ?></strong>. 
                        Follow the link in the email to set a new password.
                    </p>
                </div>

                <a href="signin.php" class="btn btn-primary" style="text-decoration: none; margin-top: 12px;">
                    <span class="btn-text">Back to Sign In</span>
                </a>

            <?php else: ?>
                <!-- Verification form -->
                <div class="info-box">
                    <strong>Identity verification required</strong><br>
                    Enter the <strong>email</strong> and <strong>first name</strong> you used when creating your account. 
                    If they match, we'll send a password reset link to your email.
                </div>

                <form method="POST" id="resetForm" novalidate>
                    <div class="form-group">
                        <label class="form-label">Email Address</label>
                        <input type="email" name="email" id="email" class="form-input"
                               placeholder="you@example.com"
                               value="<?= htmlspecialchars($_POST['email'] ?? '') ?>"
                               required autocomplete="email" autofocus>
                    </div>

                    <div class="form-group">
                        <label class="form-label">First Name (as entered during sign-up)</label>
                        <input type="text" name="first_name" id="firstName" class="form-input"
                               placeholder="Your first name"
                               value="<?= htmlspecialchars($_POST['first_name'] ?? '') ?>"
                               required autocomplete="given-name">
                    </div>

                    <button type="submit" class="btn btn-primary" id="submitBtn">
                        <div class="spinner"></div>
                        <span class="btn-text">Verify & Send Reset Link</span>
                    </button>
                </form>
            <?php endif; ?>

            <div class="auth-footer">
                <p>Remember your password? <a href="signin.php">Sign In</a></p>
                <p style="margin-top: 8px;">
                    <a href="signup.php" style="font-weight: 400; color: var(--gray-400); font-size: 13px;">Create a new account</a>
                </p>
            </div>
        </div>
    </div>
</div>

<script>
    const form = document.getElementById('resetForm');
    if (form) {
        form.addEventListener('submit', function(e) {
            const email = document.getElementById('email').value.trim();
            const firstName = document.getElementById('firstName').value.trim();

            if (!email || !firstName) {
                e.preventDefault();
                return;
            }

            const btn = document.getElementById('submitBtn');
            btn.classList.add('loading');
            btn.disabled = true;
        });
    }
</script>

<?php renderAuthFooter(); ?>
