<?php
/**
 * Dashboard — Protected by Firebase Authentication
 * Redirects unauthenticated users to sign-in page.
 */
session_start();

require_once __DIR__ . '/config/firebase_config.php';

// Auth guard: redirect to setup if Firebase not configured, or signin if not authenticated
if (!FirebaseConfig::isConfigured()) {
    header('Location: setup.php');
    exit;
}

if (empty($_SESSION['firebase_user'])) {
    header('Location: signin.php');
    exit;
}

$currentUser = $_SESSION['firebase_user'];
$displayName = htmlspecialchars($currentUser['firstName'] ?? $currentUser['email'] ?? 'User');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Analytics Dashboard</title>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        :root {
            --primary: #3b82f6;
            --primary-dark: #1e40af;
            --gray-50: #f9fafb;
            --gray-100: #f3f4f6;
            --gray-200: #e5e7eb;
            --gray-300: #d1d5db;
            --gray-600: #4b5563;
            --gray-700: #374151;
            --gray-900: #111827;
            --success: #10b981;
            --warning: #f59e0b;
            --danger: #ef4444;
        }

        /* Dark mode variables */
        body.dark-mode {
            --primary: #60a5fa;
            --gray-50: #0f172a;
            --gray-100: #1e293b;
            --gray-200: #334155;
            --gray-300: #475569;
            --gray-600: #94a3b8;
            --gray-700: #cbd5e1;
            --gray-900: #f1f5f9;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Helvetica, Arial, sans-serif;
            background: var(--gray-50);
            color: var(--gray-900);
            min-height: 100vh;
            line-height: 1.6;
        }

        /* ===== LOGIN SCREEN ===== */
        .login-container {
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            padding: 20px;
            background: linear-gradient(135deg, var(--gray-50) 0%, var(--gray-100) 100%);
        }

        .login-box {
            background: white;
            border-radius: 16px;
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
            max-width: 420px;
            width: 100%;
            padding: 48px;
        }

        .login-header {
            text-align: center;
            margin-bottom: 32px;
        }

        .login-header h1 {
            font-size: 32px;
            font-weight: 700;
            color: var(--gray-900);
            margin-bottom: 8px;
            letter-spacing: -0.5px;
        }

        .login-header p {
            color: var(--gray-600);
            font-size: 14px;
            font-weight: 500;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            display: block;
            font-size: 13px;
            font-weight: 600;
            margin-bottom: 8px;
            color: var(--gray-700);
            letter-spacing: 0.3px;
        }

        .form-group input {
            width: 100%;
            padding: 11px 14px;
            font-size: 14px;
            border: 1.5px solid var(--gray-200);
            border-radius: 8px;
            font-family: 'SF Mono', Monaco, 'Cascadia Code', 'Roboto Mono', Consolas, 'Courier New', monospace;
            transition: all 0.2s;
            background: white;
        }

        .form-group input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
            background: white;
        }

        .login-btn {
            width: 100%;
            background: var(--primary);
            color: white;
            border: none;
            padding: 11px 16px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s;
            box-shadow: 0 4px 6px -1px rgba(59, 130, 246, 0.2);
            margin-bottom: 12px;
        }

        .login-btn:hover {
            background: var(--primary-dark);
            transform: translateY(-1px);
            box-shadow: 0 10px 15px -3px rgba(59, 130, 246, 0.3);
        }

        .login-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }

        .get-key-btn {
            width: 100%;
            background: transparent;
            border: 1.5px solid var(--primary);
            color: var(--primary);
            padding: 10px 16px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s;
        }

        .get-key-btn:hover {
            background: var(--gray-50);
            border-color: var(--primary-dark);
            color: var(--primary-dark);
        }

        .error {
            background: rgba(239, 68, 68, 0.1);
            border: 1px solid rgba(239, 68, 68, 0.3);
            color: var(--danger);
            padding: 12px 14px;
            border-radius: 8px;
            margin-bottom: 16px;
            font-size: 13px;
            font-weight: 500;
        }

        .info-box {
            background: rgba(59, 130, 246, 0.05);
            border-left: 4px solid var(--primary);
            padding: 14px 16px;
            border-radius: 8px;
            font-size: 13px;
            color: var(--gray-700);
            margin-bottom: 24px;
            line-height: 1.6;
        }

        .info-box strong {
            display: block;
            margin-top: 8px;
            font-weight: 600;
            color: var(--gray-900);
        }

        .info-box code {
            background: white;
            padding: 3px 7px;
            border-radius: 4px;
            font-family: 'SF Mono', Monaco, monospace;
            display: block;
            margin-top: 6px;
            overflow-x: auto;
            word-break: break-all;
            font-size: 12px;
        }

        /* ===== DASHBOARD SCREEN ===== */
        .dashboard-container {
            display: none;
            max-width: 1600px;
            margin: 0 auto;
            padding: 24px;
        }

        .dashboard-container.active {
            display: block;
        }

        .header {
            background: white;
            padding: 28px 32px;
            border-radius: 12px;
            margin-bottom: 24px;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 20px;
        }

        .header h1 {
            color: var(--gray-900);
            font-size: 28px;
            font-weight: 700;
            letter-spacing: -0.5px;
        }

        .header p {
            color: var(--gray-600);
            font-size: 14px;
            margin-top: 4px;
        }

        .header-right {
            display: flex;
            gap: 12px;
        }

        .logout-btn, .refresh-btn {
            background: white;
            color: var(--primary);
            border: 1.5px solid var(--gray-200);
            padding: 9px 16px;
            border-radius: 8px;
            cursor: pointer;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.2s;
        }

        .logout-btn:hover, .refresh-btn:hover {
            border-color: var(--primary);
            background: var(--gray-50);
            transform: translateY(-1px);
        }

        .theme-toggle-btn {
            background: white;
            color: var(--primary);
            border: 1.5px solid var(--gray-200);
            padding: 9px 16px;
            border-radius: 8px;
            cursor: pointer;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.2s;
        }

        .theme-toggle-btn:hover {
            border-color: var(--primary);
            background: var(--gray-50);
            transform: translateY(-1px);
        }

        body.dark-mode .logout-btn,
        body.dark-mode .refresh-btn,
        body.dark-mode .theme-toggle-btn {
            background: var(--gray-100);
            color: var(--primary);
            border-color: var(--gray-300);
        }

        body.dark-mode .logout-btn:hover,
        body.dark-mode .refresh-btn:hover,
        body.dark-mode .theme-toggle-btn:hover {
            background: var(--gray-200);
        }

        .api-info {
            background: rgba(59, 130, 246, 0.08);
            border-left: 4px solid var(--primary);
            padding: 12px 16px;
            border-radius: 8px;
            font-size: 13px;
            color: var(--gray-700);
            margin-bottom: 20px;
        }

        .api-info code {
            background: white;
            padding: 2px 6px;
            border-radius: 4px;
            font-family: 'SF Mono', Monaco, monospace;
            font-size: 12px;
        }

        .tabs {
            display: flex;
            gap: 8px;
            margin-bottom: 28px;
            background: white;
            padding: 6px;
            border-radius: 10px;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
            border-bottom: 1px solid var(--gray-200);
        }

        .tab-btn {
            background: transparent;
            border: none;
            padding: 10px 16px;
            border-radius: 8px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            color: var(--gray-600);
            transition: all 0.2s ease;
            white-space: nowrap;
        }

        .tab-btn.active {
            background: white;
            color: var(--primary);
            box-shadow: 0 1px 3px rgba(59, 130, 246, 0.1);
        }

        .tab-btn:hover:not(.active) {
            color: var(--primary);
            background: var(--gray-50);
        }

        .section {
            display: none;
        }

        .section.active {
            display: block;
            animation: fadeIn 0.2s ease;
        }

        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
            gap: 16px;
            margin-bottom: 24px;
        }

        .stat-card {
            background: white;
            padding: 24px;
            border-radius: 12px;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
            border-top: 3px solid var(--primary);
            transition: all 0.3s;
        }

        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
        }

        .stat-card h3 {
            color: var(--gray-600);
            font-size: 12px;
            margin-bottom: 12px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .stat-card .value {
            font-size: 32px;
            font-weight: 700;
            color: var(--gray-900);
            letter-spacing: -1px;
        }

        .charts-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(340px, 1fr));
            gap: 16px;
            margin-bottom: 24px;
        }

        .chart-card, .list-card, .table-card {
            background: white;
            padding: 24px;
            border-radius: 12px;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
        }

        .chart-card h3, .list-card h3, .table-card h3 {
            color: var(--gray-900);
            margin-bottom: 18px;
            font-size: 15px;
            font-weight: 700;
            letter-spacing: -0.3px;
        }

        .list-group {
            list-style: none;
        }

        .list-group li {
            padding: 12px 0;
            border-bottom: 1px solid var(--gray-100);
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-size: 13px;
        }

        .list-group li:last-child {
            border-bottom: none;
        }

        .badge {
            background: var(--primary);
            color: white;
            padding: 5px 10px;
            border-radius: 6px;
            font-size: 12px;
            font-weight: 700;
        }

        .table-card {
            margin-bottom: 16px;
            overflow-x: auto;
        }

        table {
            width: 100%;
            border-collapse: collapse;
            font-size: 13px;
        }

        table thead {
            background: var(--gray-50);
            border-bottom: 2px solid var(--gray-200);
        }

        table th {
            padding: 12px 14px;
            text-align: left;
            color: var(--gray-900);
            font-weight: 700;
            font-size: 12px;
            text-transform: uppercase;
            letter-spacing: 0.3px;
        }

        table td {
            padding: 13px 14px;
            border-bottom: 1px solid var(--gray-100);
            color: var(--gray-700);
        }

        table tr:hover {
            background: var(--gray-50);
        }

        table code {
            background: var(--gray-100);
            padding: 3px 7px;
            border-radius: 4px;
            font-family: 'SF Mono', Monaco, monospace;
            font-size: 12px;
            color: var(--gray-700);
        }

        body.dark-mode table thead {
            background: var(--gray-200);
            border-bottom-color: var(--gray-300);
        }

        body.dark-mode table td {
            border-bottom-color: var(--gray-300);
        }

        body.dark-mode table tr:hover {
            background: var(--gray-200);
        }

        body.dark-mode table code {
            background: var(--gray-300);
            color: var(--gray-700);
        }

        .loading {
            text-align: center;
            padding: 60px 20px;
            color: var(--gray-600);
            font-size: 14px;
        }

        .device-config, .country-config {
            background: white;
            padding: 24px;
            border-radius: 12px;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
            margin-bottom: 16px;
        }

        .device-config h3, .country-config h3 {
            font-size: 16px;
            font-weight: 700;
            color: var(--gray-900);
            margin-bottom: 20px;
            letter-spacing: -0.3px;
        }

        .device-controls {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(140px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }

        .device-control {
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .device-control input[type="checkbox"] {
            width: 20px;
            height: 20px;
            cursor: pointer;
            accent-color: var(--primary);
        }

        .device-control label {
            cursor: pointer;
            color: var(--gray-700);
            font-size: 14px;
            font-weight: 500;
        }

        .form-group-dashboard {
            margin-bottom: 18px;
        }

        .form-group-dashboard label {
            display: block;
            font-size: 13px;
            font-weight: 600;
            margin-bottom: 8px;
            color: var(--gray-900);
        }

        textarea, select {
            width: 100%;
            padding: 11px 14px;
            font-size: 13px;
            border: 1.5px solid var(--gray-200);
            border-radius: 8px;
            font-family: 'SF Mono', Monaco, 'Cascadia Code', monospace;
            transition: all 0.2s;
            background: white;
            color: var(--gray-900);
        }

        textarea:focus, select:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
            background: white;
        }

        textarea {
            min-height: 100px;
            resize: vertical;
        }

        .save-btn {
            background: var(--primary);
            color: white;
            border: none;
            padding: 10px 18px;
            border-radius: 8px;
            cursor: pointer;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.2s;
            box-shadow: 0 2px 4px rgba(59, 130, 246, 0.2);
        }

        .save-btn:hover {
            background: var(--primary-dark);
            transform: translateY(-1px);
            box-shadow: 0 4px 8px rgba(59, 130, 246, 0.3);
        }

        .save-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }

        @media (max-width: 768px) {
            .dashboard-container { padding: 16px; }
            .login-box { padding: 32px 24px; }
            .header { flex-direction: column; align-items: flex-start; padding: 20px 24px; }
            .header-right { width: 100%; gap: 8px; }
            .logout-btn, .refresh-btn { flex: 1; }
            .stats-grid { grid-template-columns: repeat(2, 1fr); gap: 12px; }
            .charts-grid { grid-template-columns: 1fr; }
            .device-controls { grid-template-columns: 1fr; }
            .tabs { flex-wrap: wrap; }
            table { font-size: 12px; }
            table th, table td { padding: 10px 8px; }
        }

        /* Dark mode support for all elements */
        body.dark-mode {
            background: var(--gray-50);
            color: var(--gray-900);
        }

        body.dark-mode .login-container {
            background: linear-gradient(135deg, var(--gray-50) 0%, var(--gray-100) 100%);
        }

        body.dark-mode .login-box,
        body.dark-mode .header,
        body.dark-mode .tabs,
        body.dark-mode .stat-card,
        body.dark-mode .chart-card,
        body.dark-mode .list-card,
        body.dark-mode .table-card,
        body.dark-mode .device-config,
        body.dark-mode .country-config {
            background: var(--gray-100);
            color: var(--gray-900);
        }

        body.dark-mode .stat-card {
            border-top-color: var(--primary);
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.3);
        }

        body.dark-mode .stat-card:hover {
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
        }

        body.dark-mode .chart-card,
        body.dark-mode .list-card,
        body.dark-mode .table-card,
        body.dark-mode .device-config,
        body.dark-mode .country-config {
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.3);
        }

        body.dark-mode .list-group li {
            border-bottom-color: var(--gray-200);
        }

        body.dark-mode .list-group li:last-child {
            border-bottom: none;
        }

        body.dark-mode .form-group input,
        body.dark-mode textarea,
        body.dark-mode select {
            background: var(--gray-200);
            color: var(--gray-900);
            border-color: var(--gray-300);
        }

        body.dark-mode .form-group input:focus,
        body.dark-mode textarea:focus,
        body.dark-mode select:focus {
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(96, 165, 250, 0.1);
        }

        body.dark-mode .info-box {
            background: rgba(96, 165, 250, 0.1);
            border-left-color: var(--primary);
            color: var(--gray-700);
        }

        body.dark-mode .info-box strong {
            color: var(--gray-900);
        }

        body.dark-mode .info-box code {
            background: var(--gray-200);
            color: var(--gray-900);
        }

        body.dark-mode .error {
            background: rgba(239, 68, 68, 0.15);
            border-color: rgba(239, 68, 68, 0.4);
            color: #ff6b6b;
        }

        body.dark-mode .api-info {
            background: rgba(96, 165, 250, 0.1);
            border-left-color: var(--primary);
            color: var(--gray-700);
        }

        body.dark-mode .api-info code {
            background: var(--gray-200);
            color: var(--gray-900);
        }

        body.dark-mode .header {
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.3);
        }

        body.dark-mode .header p {
            color: var(--gray-600);
        }
    </style>
</head>
<body>
    <!-- LOGIN SCREEN -->
    <div class="login-container" id="loginContainer">
        <div class="login-box">
            <div class="login-header">
                <h1>Analytics</h1>
                <p>Dashboard Login</p>
            </div>

            <div id="loginError" class="error" style="display: none;"></div>

            <div class="info-box">
                <strong>Step 1: Get your API key</strong>
                Visit your redirect hosting and open api.php:
                <code id="keyGenUrl">https://yourdomain.com/redirect/api.php</code>
                <br><br>
                <strong>Step 2: Enter credentials below</strong>
            </div>

            <form id="loginForm" onsubmit="handleLogin(event)">
                <div class="form-group">
                    <label>API URL</label>
                    <input type="url" id="apiUrl" placeholder="https://yourdomain.com/redirect/api.php" required>
                </div>

                <div class="form-group">
                    <label>API Key</label>
                    <input type="text" id="apiKey" placeholder="10-digit API key" maxlength="10" required>
                </div>

                <button type="submit" class="login-btn">Sign In</button>
            </form>

            <button type="button" class="get-key-btn" onclick="openGetKeyWindow()">
                Generate New Key
            </button>
        </div>
    </div>

    <!-- DASHBOARD SCREEN -->
    <div class="dashboard-container" id="dashboardContainer">
        <div class="header">
            <div>
                <h1>Analytics</h1>
                <p id="connectedHost">Connected to: <strong></strong></p>
            </div>
            <div class="header-right">
                <span style="display: flex; align-items: center; gap: 8px; font-size: 13px; color: var(--gray-600); font-weight: 500; padding: 0 8px;">
                    👤 <?= $displayName ?>
                </span>
                <button class="theme-toggle-btn" onclick="toggleDarkMode()" title="Toggle dark mode">🌙</button>
                <button class="refresh-btn" onclick="location.reload()">Refresh</button>
                <button class="logout-btn" onclick="handleLogout()">Logout</button>
            </div>
        </div>

        <div class="api-info">
            Endpoint: <code id="displayApiUrl"></code>
        </div>

        <div class="tabs">
            <button class="tab-btn active" onclick="switchTab('visitors', event)">Visitors</button>
            <button class="tab-btn" onclick="switchTab('blocked', event)">Blocked</button>
            <button class="tab-btn" onclick="switchTab('settings', event)">Settings</button>
        </div>

        <!-- VISITORS TAB -->
        <div id="visitors" class="section active">
            <div class="stats-grid">
                <div class="stat-card">
                    <h3>Total Visitors</h3>
                    <div class="value" id="totalVisitors">0</div>
                </div>
                <div class="stat-card">
                    <h3>Countries</h3>
                    <div class="value" id="totalCountries">0</div>
                </div>
                <div class="stat-card">
                    <h3>Browsers</h3>
                    <div class="value" id="totalBrowsers">0</div>
                </div>
                <div class="stat-card">
                    <h3>Operating Systems</h3>
                    <div class="value" id="totalOS">0</div>
                </div>
            </div>

            <div class="charts-grid">
                <div class="chart-card">
                    <h3>Top Browsers</h3>
                    <canvas id="browserChart"></canvas>
                </div>
                <div class="chart-card">
                    <h3>Operating Systems</h3>
                    <canvas id="osChart"></canvas>
                </div>
                <div class="chart-card">
                    <h3>Device Types</h3>
                    <canvas id="deviceChart"></canvas>
                </div>
            </div>

            <div class="charts-grid">
                <div class="list-card">
                    <h3>Top Countries</h3>
                    <ul class="list-group" id="countriesList">
                        <li><em>Loading...</em></li>
                    </ul>
                </div>
                <div class="list-card">
                    <h3>Top States</h3>
                    <ul class="list-group" id="statesList">
                        <li><em>Loading...</em></li>
                    </ul>
                </div>
            </div>

            <div class="table-card">
                <h3>Recent Visitors</h3>
                <div id="visitorsTable" class="loading">Loading...</div>
            </div>
        </div>

        <!-- BLOCKED BOTS TAB -->
        <div id="blocked" class="section">
            <div class="stats-grid">
                <div class="stat-card">
                    <h3>Total Blocked</h3>
                    <div class="value" id="totalBlocked">0</div>
                </div>
                <div class="stat-card">
                    <h3>Block Reasons</h3>
                    <div class="value" id="totalReasons">0</div>
                </div>
                <div class="stat-card">
                    <h3>Blocked Countries</h3>
                    <div class="value" id="blockedCountries">0</div>
                </div>
            </div>

            <div class="charts-grid">
                <div class="list-card">
                    <h3>Block Reasons</h3>
                    <ul class="list-group" id="reasonsList">
                        <li><em>Loading...</em></li>
                    </ul>
                </div>
                <div class="list-card">
                    <h3>Blocked Countries</h3>
                    <ul class="list-group" id="blockedCountriesList">
                        <li><em>Loading...</em></li>
                    </ul>
                </div>
            </div>

            <div class="table-card">
                <h3>Recent Blocked Attempts</h3>
                <div id="blockedBotsTable" class="loading">Loading...</div>
            </div>
        </div>

        <!-- SETTINGS TAB -->
        <div id="settings" class="section">
            <div class="device-config">
                <h3>Device Access Control</h3>
                <form id="deviceForm" onsubmit="handleUpdateDevices(event)">
                    <div class="device-controls">
                        <div class="device-control">
                            <input type="checkbox" id="allow_desktop" checked>
                            <label for="allow_desktop">Desktop</label>
                        </div>
                        <div class="device-control">
                            <input type="checkbox" id="allow_mobile" checked>
                            <label for="allow_mobile">Mobile</label>
                        </div>
                        <div class="device-control">
                            <input type="checkbox" id="allow_tablet" checked>
                            <label for="allow_tablet">Tablet</label>
                        </div>
                    </div>
                    <button type="submit" class="save-btn">Save Changes</button>
                </form>
            </div>

            <div class="country-config">
                <h3>Country Access Control</h3>
                <form id="countryForm" onsubmit="handleUpdateCountries(event)">
                    <div class="form-group-dashboard">
                        <label for="country_mode">Access Mode</label>
                        <select id="country_mode">
                            <option value="allow_all">Allow All Countries</option>
                            <option value="whitelist">Whitelist (Only listed)</option>
                            <option value="blacklist">Blacklist (Except listed)</option>
                        </select>
                    </div>

                    <div class="form-group-dashboard">
                        <label for="countries_list">Country Codes</label>
                        <textarea id="countries_list" placeholder="e.g., US, CA, GB, AU"></textarea>
                    </div>

                    <button type="submit" class="save-btn">Save Changes</button>
                </form>
            </div>

            <div class="country-config">
                <h3>Redirect Destination</h3>
                <form id="redirectForm" onsubmit="handleUpdateRedirectUrl(event)">
                    <div class="form-group-dashboard">
                        <label for="redirect_url">Destination URL</label>
                        <textarea id="redirect_url" placeholder="https://example.com/destination"></textarea>
                    </div>

                    <div style="background: rgba(59, 130, 246, 0.05); border-left: 4px solid var(--primary); padding: 14px 16px; border-radius: 8px; font-size: 13px; color: var(--gray-700); margin-bottom: 18px; line-height: 1.6;">
                        <strong style="color: var(--gray-900); display: block; margin-bottom: 6px;">Note:</strong> 
                        Visitor emails (if provided) will be automatically appended as a hash parameter.
                        <br><code style="background: white; padding: 4px 8px; border-radius: 4px; display: block; margin-top: 8px; font-size: 12px;">https://example.com/destination#visitor@email.com</code>
                    </div>

                    <button type="submit" class="save-btn">Save Changes</button>
                </form>
            </div>
        </div>
    </div>

    <script>
        let apiConfig = { url: null, key: null };
        let charts = {};

        function getStoredCredentials() {
            const stored = localStorage.getItem('dashboardCredentials');
            return stored ? JSON.parse(stored) : null;
        }

        function saveCredentials(url, key) {
            localStorage.setItem('dashboardCredentials', JSON.stringify({ url, key }));
        }

        async function callApi(action, params = {}, method = 'GET') {
            if (!apiConfig.url || !apiConfig.key) {
                throw new Error('Not connected');
            }

            let url = apiConfig.url + '?action=' + action;
            const headers = { 'X-API-Key': apiConfig.key };

            if (method === 'GET' && Object.keys(params).length > 0) {
                url += '&' + new URLSearchParams(params).toString();
            }

            const options = { method, headers };
            if (method === 'POST') {
                headers['Content-Type'] = 'application/json';
                options.body = JSON.stringify(params);
            }

            const response = await fetch(url, options).catch(e => {
                throw new Error('Network error: ' + e.message);
            });

            const data = await response.json().catch(e => {
                throw new Error('Invalid response from server');
            });

            if (!response.ok || data.error) {
                throw new Error(data.error || data.message || 'API error');
            }

            return data;
        }

        async function handleLogin(event) {
            event.preventDefault();
            const btn = event.target.querySelector('.login-btn');
            const errorDiv = document.getElementById('loginError');

            btn.disabled = true;
            btn.textContent = 'Signing in...';
            errorDiv.style.display = 'none';

            try {
                const url = document.getElementById('apiUrl').value.trim();
                const key = document.getElementById('apiKey').value.trim();

                apiConfig = { url, key };
                await callApi('verify_credentials', { api_key: key }, 'POST');

                saveCredentials(url, key);
                showDashboard(url);
                await loadAllData();
            } catch (error) {
                errorDiv.textContent = error.message;
                errorDiv.style.display = 'block';
                btn.disabled = false;
                btn.textContent = 'Sign In';
            }
        }

        function showDashboard(apiUrl) {
            document.getElementById('loginContainer').style.display = 'none';
            document.getElementById('dashboardContainer').classList.add('active');
            document.getElementById('displayApiUrl').textContent = apiUrl;
            const host = new URL(apiUrl).host;
            document.getElementById('connectedHost').querySelector('strong').textContent = host;
        }

        function handleLogout() {
            if (confirm('Sign out and disconnect?')) {
                localStorage.removeItem('dashboardCredentials');
                window.location.href = 'logout.php';
            }
        }

        async function loadAllData() {
            try {
                const [stats, botStats, config, visitors, bots, redirectData] = await Promise.all([
                    callApi('get_visitor_stats'),
                    callApi('get_blocked_bot_stats'),
                    callApi('get_config'),
                    callApi('get_visitors', { limit: 15, offset: 0 }),
                    callApi('get_blocked_bots', { limit: 15, offset: 0 }),
                    callApi('get_redirect_url')
                ]);

                updateStatistics(stats.data);
                updateBotStatistics(botStats.data);
                updateConfiguration(config.data, redirectData.redirect_url);
                renderVisitorsTable(visitors.data);
                renderBotsTable(bots.data);
                initializeCharts(stats.data);
            } catch (error) {
                console.error('Load error:', error);
            }
        }

        function updateStatistics(stats) {
            document.getElementById('totalVisitors').textContent = (stats.total_visitors || 0).toLocaleString();
            document.getElementById('totalCountries').textContent = (stats.by_country || []).length;
            document.getElementById('totalBrowsers').textContent = (stats.by_browser || []).length;
            document.getElementById('totalOS').textContent = (stats.by_os || []).length;

            document.getElementById('countriesList').innerHTML = (stats.by_country || [])
                .slice(0, 10)
                .map(item => `<li><span>${esc(item.country)}</span><span class="badge">${item.count}</span></li>`)
                .join('') || '<li><em>No data</em></li>';

            document.getElementById('statesList').innerHTML = (stats.by_state || [])
                .slice(0, 10)
                .map(item => `<li><span>${esc(item.state)}</span><span class="badge">${item.count}</span></li>`)
                .join('') || '<li><em>No data</em></li>';
        }

        function updateBotStatistics(stats) {
            document.getElementById('totalBlocked').textContent = (stats.total || 0).toLocaleString();
            document.getElementById('totalReasons').textContent = (stats.by_reason || []).length;
            document.getElementById('blockedCountries').textContent = (stats.by_country || []).length;

            document.getElementById('reasonsList').innerHTML = (stats.by_reason || [])
                .slice(0, 15)
                .map(item => `<li><span>${esc(item.reason.replace(/_/g, ' '))}</span><span class="badge">${item.count}</span></li>`)
                .join('') || '<li><em>No data</em></li>';

            document.getElementById('blockedCountriesList').innerHTML = (stats.by_country || [])
                .slice(0, 15)
                .map(item => `<li><span>${esc(item.country)}</span><span class="badge">${item.count}</span></li>`)
                .join('') || '<li><em>No data</em></li>';
        }

        function updateConfiguration(config, redirectUrl) {
            document.getElementById('allow_desktop').checked = config.allow_desktop;
            document.getElementById('allow_mobile').checked = config.allow_mobile;
            document.getElementById('allow_tablet').checked = config.allow_tablet;
            document.getElementById('country_mode').value = config.block_mode || 'allow_all';

            const countries = config.block_mode === 'whitelist' ? config.allowed_countries :
                              config.block_mode === 'blacklist' ? config.blocked_countries : [];
            document.getElementById('countries_list').value = (countries || []).join(', ');
            
            document.getElementById('redirect_url').value = redirectUrl || 'https://example.com/destination';
        }

        function renderVisitorsTable(visitors) {
            if (!visitors || visitors.length === 0) {
                document.getElementById('visitorsTable').innerHTML = '<p style="text-align:center; color: var(--gray-600); padding: 40px 20px;">No visitor data available</p>';
                return;
            }

            const html = `
                <table>
                    <thead>
                        <tr>
                            <th style="width: 140px;">Date & Time</th>
                            <th style="width: 120px;">IP Address</th>
                            <th style="width: 80px;">Country</th>
                            <th style="width: 150px;">Email</th>
                            <th style="width: 100px;">Browser</th>
                            <th style="width: 100px;">OS</th>
                            <th style="width: 80px;">Device</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${visitors.map(v => `
                            <tr>
                                <td>${formatDate(v.timestamp)}</td>
                                <td><code>${esc(v.ip_address)}</code></td>
                                <td>${esc(v.country || '—')}</td>
                                <td>${v.email ? `<code style="background: transparent; padding: 0;">${esc(v.email.substring(0, 30))}</code>` : '—'}</td>
                                <td>${esc((v.browser || 'Unknown').substring(0, 20))}</td>
                                <td>${esc(v.os || 'Unknown')}</td>
                                <td><span class="badge">${esc((v.device_type || 'Unknown').substring(0, 10))}</span></td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            `;
            document.getElementById('visitorsTable').innerHTML = html;
        }

        function renderBotsTable(bots) {
            if (!bots || bots.length === 0) {
                document.getElementById('blockedBotsTable').innerHTML = '<p style="text-align:center; color: var(--gray-600); padding: 40px 20px;">No blocked attempts recorded</p>';
                return;
            }

            const html = `
                <table>
                    <thead>
                        <tr>
                            <th style="width: 140px;">Date & Time</th>
                            <th style="width: 120px;">IP Address</th>
                            <th style="width: 80px;">Country</th>
                            <th style="width: 140px;">Block Reason</th>
                            <th style="width: 100px;">Browser</th>
                            <th style="width: 100px;">OS</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${bots.map(b => `
                            <tr>
                                <td>${formatDate(b.timestamp)}</td>
                                <td><code>${esc(b.ip_address)}</code></td>
                                <td>${esc(b.country || '—')}</td>
                                <td>${esc(b.reason.replace(/_/g, ' ').substring(0, 25))}</td>
                                <td>${esc((b.browser || 'Unknown').substring(0, 20))}</td>
                                <td>${esc(b.os || 'Unknown')}</td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            `;
            document.getElementById('blockedBotsTable').innerHTML = html;
        }

        async function handleUpdateDevices(event) {
            event.preventDefault();
            const btn = event.target.querySelector('.save-btn');
            btn.disabled = true;
            btn.textContent = 'Saving...';

            try {
                await callApi('update_device_settings', {
                    allow_desktop: document.getElementById('allow_desktop').checked,
                    allow_mobile: document.getElementById('allow_mobile').checked,
                    allow_tablet: document.getElementById('allow_tablet').checked
                }, 'POST');
                alert('Changes saved successfully');
            } catch (error) {
                alert('Error: ' + error.message);
            } finally {
                btn.disabled = false;
                btn.textContent = 'Save Changes';
            }
        }

        async function handleUpdateCountries(event) {
            event.preventDefault();
            const btn = event.target.querySelector('.save-btn');
            btn.disabled = true;
            btn.textContent = 'Saving...';

            try {
                const countriesText = document.getElementById('countries_list').value;
                const countries = countriesText ?
                    countriesText.split(',').map(c => c.trim().toUpperCase()).filter(c => c.length === 2) : [];

                await callApi('update_country_settings', {
                    block_mode: document.getElementById('country_mode').value,
                    countries: countries
                }, 'POST');
                alert('Changes saved successfully');
            } catch (error) {
                alert('Error: ' + error.message);
            } finally {
                btn.disabled = false;
                btn.textContent = 'Save Changes';
            }
        }

        async function handleUpdateRedirectUrl(event) {
            event.preventDefault();
            const btn = event.target.querySelector('.save-btn');
            btn.disabled = true;
            btn.textContent = 'Saving...';

            try {
                const redirectUrl = document.getElementById('redirect_url').value.trim();
                
                if (!redirectUrl) {
                    throw new Error('Destination URL cannot be empty');
                }

                await callApi('update_redirect_url', {
                    redirect_url: redirectUrl
                }, 'POST');
                alert('Changes saved successfully');
            } catch (error) {
                alert('Error: ' + error.message);
            } finally {
                btn.disabled = false;
                btn.textContent = 'Save Changes';
            }
        }

        function switchTab(tabId, event) {
            document.querySelectorAll('.section').forEach(el => el.classList.remove('active'));
            document.getElementById(tabId).classList.add('active');
            document.querySelectorAll('.tab-btn').forEach(btn => btn.classList.remove('active'));
            event.target.classList.add('active');
        }

        function initializeCharts(stats) {
            const opts = {
                responsive: true,
                maintainAspectRatio: true,
                plugins: { 
                    legend: { 
                        position: 'bottom', 
                        labels: { 
                            font: { size: 12, family: "'-apple-system, BlinkMacSystemFont, Segoe UI" },
                            color: 'var(--gray-700)',
                            padding: 15
                        },
                        align: 'center'
                    } 
                }
            };

            if ((stats.by_browser || []).length > 0) {
                if (charts.browser) charts.browser.destroy();
                charts.browser = new Chart(document.getElementById('browserChart').getContext('2d'), {
                    type: 'doughnut',
                    data: {
                        labels: stats.by_browser.slice(0, 6).map(b => b.browser),
                        datasets: [{ 
                            data: stats.by_browser.slice(0, 6).map(b => b.count),
                            backgroundColor: ['#3b82f6', '#ec4899', '#f59e0b', '#10b981', '#6366f1', '#8b5cf6'],
                            borderColor: 'white',
                            borderWidth: 2
                        }]
                    },
                    options: opts
                });
            }

            if ((stats.by_os || []).length > 0) {
                if (charts.os) charts.os.destroy();
                charts.os = new Chart(document.getElementById('osChart').getContext('2d'), {
                    type: 'doughnut',
                    data: {
                        labels: stats.by_os.slice(0, 6).map(o => o.os),
                        datasets: [{ 
                            data: stats.by_os.slice(0, 6).map(o => o.count),
                            backgroundColor: ['#3b82f6', '#ec4899', '#f59e0b', '#10b981', '#6366f1', '#8b5cf6'],
                            borderColor: 'white',
                            borderWidth: 2
                        }]
                    },
                    options: opts
                });
            }

            if ((stats.by_device || []).length > 0) {
                if (charts.device) charts.device.destroy();
                charts.device = new Chart(document.getElementById('deviceChart').getContext('2d'), {
                    type: 'bar',
                    data: {
                        labels: stats.by_device.map(d => d.device_type),
                        datasets: [{ 
                            label: 'Count', 
                            data: stats.by_device.map(d => d.count), 
                            backgroundColor: '#3b82f6',
                            borderRadius: 6
                        }]
                    },
                    options: { 
                        ...opts, 
                        plugins: { 
                            ...opts.plugins, 
                            legend: { display: false } 
                        },
                        scales: {
                            y: {
                                beginAtZero: true,
                                ticks: { color: 'var(--gray-700)' },
                                grid: { color: 'var(--gray-100)' }
                            },
                            x: {
                                ticks: { color: 'var(--gray-700)' },
                                grid: { display: false }
                            }
                        }
                    }
                });
            }
        }

        function formatDate(date) {
            try {
                return new Date(date).toLocaleDateString('en-US', {
                    month: 'short', day: 'numeric', year: 'numeric',
                    hour: '2-digit', minute: '2-digit'
                });
            } catch (e) {
                return date;
            }
        }

        function esc(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        function openGetKeyWindow() {
            const url = document.getElementById('apiUrl').value.trim();
            window.open(url || 'https://yourdomain.com/redirect/api.php', '_blank');
        }

        window.addEventListener('DOMContentLoaded', function() {
            const creds = getStoredCredentials();
            if (creds) {
                apiConfig = creds;
                showDashboard(creds.url);
                loadAllData();
            }

            // Initialize dark mode from localStorage
            const savedDarkMode = localStorage.getItem('dashboardDarkMode');
            if (savedDarkMode === 'true') {
                document.body.classList.add('dark-mode');
                updateThemeToggleButton();
            }
        });

        function toggleDarkMode() {
            document.body.classList.toggle('dark-mode');
            const isDarkMode = document.body.classList.contains('dark-mode');
            localStorage.setItem('dashboardDarkMode', isDarkMode);
            updateThemeToggleButton();
        }

        function updateThemeToggleButton() {
            const btn = document.querySelector('.theme-toggle-btn');
            if (btn) {
                const isDarkMode = document.body.classList.contains('dark-mode');
                btn.textContent = isDarkMode ? '☀️' : '🌙';
            }
        }
    </script>
</body>
</html>
