<?php
/**
 * Firebase Configuration Manager
 * Stores and retrieves Firebase credentials securely via local encrypted file.
 * Credentials are never hardcoded - they are entered via the setup page.
 */

class FirebaseConfig {
    private static $configFile;
    private static $encryptionKey;

    public static function init() {
        self::$configFile = __DIR__ . '/.firebase_credentials.dat';
        // Derive encryption key from server-specific values
        self::$encryptionKey = hash('sha256', __DIR__ . php_uname('n') . 'firebase_salt_key', true);
    }

    /**
     * Check if Firebase credentials have been configured
     */
    public static function isConfigured(): bool {
        self::init();
        return file_exists(self::$configFile);
    }

    /**
     * Save Firebase credentials (encrypted)
     */
    public static function saveCredentials(array $credentials): bool {
        self::init();
        
        $required = ['api_key', 'auth_domain', 'project_id'];
        foreach ($required as $field) {
            if (empty($credentials[$field])) {
                return false;
            }
        }

        $data = json_encode([
            'api_key'     => trim($credentials['api_key']),
            'auth_domain' => trim($credentials['auth_domain']),
            'project_id'  => trim($credentials['project_id']),
            'created_at'  => date('Y-m-d H:i:s'),
            'updated_at'  => date('Y-m-d H:i:s')
        ]);

        $iv = openssl_random_pseudo_bytes(16);
        $encrypted = openssl_encrypt($data, 'AES-256-CBC', self::$encryptionKey, 0, $iv);
        $stored = base64_encode($iv . '::' . $encrypted);

        return file_put_contents(self::$configFile, $stored) !== false;
    }

    /**
     * Get Firebase credentials (decrypted)
     */
    public static function getCredentials(): ?array {
        self::init();

        if (!self::isConfigured()) {
            return null;
        }

        $stored = file_get_contents(self::$configFile);
        $decoded = base64_decode($stored);
        $parts = explode('::', $decoded, 2);

        if (count($parts) !== 2) {
            return null;
        }

        $iv = $parts[0];
        $encrypted = $parts[1];
        $decrypted = openssl_decrypt($encrypted, 'AES-256-CBC', self::$encryptionKey, 0, $iv);

        if ($decrypted === false) {
            return null;
        }

        return json_decode($decrypted, true);
    }

    /**
     * Get the Firebase Web API Key
     */
    public static function getApiKey(): ?string {
        $creds = self::getCredentials();
        return $creds['api_key'] ?? null;
    }

    /**
     * Delete stored credentials
     */
    public static function deleteCredentials(): bool {
        self::init();
        if (file_exists(self::$configFile)) {
            return unlink(self::$configFile);
        }
        return true;
    }

    /**
     * Update credentials (preserves created_at)
     */
    public static function updateCredentials(array $newCredentials): bool {
        self::init();
        $existing = self::getCredentials();
        
        $merged = array_merge($existing ?? [], $newCredentials);
        $merged['updated_at'] = date('Y-m-d H:i:s');
        
        if ($existing) {
            $merged['created_at'] = $existing['created_at'];
        }

        $data = json_encode($merged);
        $iv = openssl_random_pseudo_bytes(16);
        $encrypted = openssl_encrypt($data, 'AES-256-CBC', self::$encryptionKey, 0, $iv);
        $stored = base64_encode($iv . '::' . $encrypted);

        return file_put_contents(self::$configFile, $stored) !== false;
    }
}
