<?php
/**
 * Local User Profile Store
 * Stores email → first name mappings for password recovery verification.
 * Data is encrypted at rest — no plaintext credentials on disk.
 */

class UserStore {
    private static $storeFile;
    private static $encryptionKey;

    public static function init() {
        self::$storeFile = __DIR__ . '/../../config/.user_profiles.dat';
        self::$encryptionKey = hash('sha256', __DIR__ . php_uname('n') . 'user_store_salt', true);
    }

    /**
     * Save a user profile (email → first name)
     */
    public static function saveUser(string $email, string $firstName): bool {
        self::init();
        $users = self::getAllUsers();
        $users[strtolower(trim($email))] = [
            'first_name' => trim($firstName),
            'created_at' => date('Y-m-d H:i:s'),
        ];
        return self::writeStore($users);
    }

    /**
     * Get the first name for a given email
     */
    public static function getFirstName(string $email): ?string {
        self::init();
        $users = self::getAllUsers();
        $key = strtolower(trim($email));
        return $users[$key]['first_name'] ?? null;
    }

    /**
     * Verify that the first name matches the stored one for an email
     */
    public static function verifyFirstName(string $email, string $firstName): bool {
        $stored = self::getFirstName($email);
        if ($stored === null) {
            return false;
        }
        return strtolower(trim($stored)) === strtolower(trim($firstName));
    }

    /**
     * Check if a user exists
     */
    public static function userExists(string $email): bool {
        self::init();
        $users = self::getAllUsers();
        return isset($users[strtolower(trim($email))]);
    }

    /**
     * Delete a user profile
     */
    public static function deleteUser(string $email): bool {
        self::init();
        $users = self::getAllUsers();
        unset($users[strtolower(trim($email))]);
        return self::writeStore($users);
    }

    /**
     * Get all stored users (decrypted)
     */
    private static function getAllUsers(): array {
        if (!file_exists(self::$storeFile)) {
            return [];
        }

        $stored = file_get_contents(self::$storeFile);
        if (empty($stored)) {
            return [];
        }

        $decoded = base64_decode($stored);
        $parts = explode('::', $decoded, 2);

        if (count($parts) !== 2) {
            return [];
        }

        $decrypted = openssl_decrypt($parts[1], 'AES-256-CBC', self::$encryptionKey, 0, $parts[0]);
        if ($decrypted === false) {
            return [];
        }

        return json_decode($decrypted, true) ?: [];
    }

    /**
     * Write the user store (encrypted)
     */
    private static function writeStore(array $users): bool {
        $data = json_encode($users);
        $iv = openssl_random_pseudo_bytes(16);
        $encrypted = openssl_encrypt($data, 'AES-256-CBC', self::$encryptionKey, 0, $iv);
        $stored = base64_encode($iv . '::' . $encrypted);

        $dir = dirname(self::$storeFile);
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }

        return file_put_contents(self::$storeFile, $stored) !== false;
    }
}
