<?php
/**
 * Firebase REST API Authentication Handler
 * Uses Firebase Identity Toolkit REST API — no SDK/CDN required.
 */

require_once __DIR__ . '/../../config/firebase_config.php';

class FirebaseAuth {
    private const BASE_URL = 'https://identitytoolkit.googleapis.com/v1/accounts';
    private $apiKey;

    public function __construct() {
        $creds = FirebaseConfig::getCredentials();
        if (!$creds || empty($creds['api_key'])) {
            throw new Exception('Firebase credentials not configured. Please run setup first.');
        }
        $this->apiKey = $creds['api_key'];
    }

    /**
     * Sign up a new user with email, password, and display name (first name)
     */
    public function signUp(string $email, string $password, string $firstName): array {
        // Step 1: Create the account
        $response = $this->request('signUp', [
            'email'             => $email,
            'password'          => $password,
            'returnSecureToken' => true,
        ]);

        if (isset($response['error'])) {
            return ['success' => false, 'error' => $this->parseError($response['error'])];
        }

        // Step 2: Update the profile with the first name as displayName
        $this->request('update', [
            'idToken'     => $response['idToken'],
            'displayName' => $firstName,
        ]);

        return [
            'success'      => true,
            'uid'          => $response['localId'],
            'email'        => $response['email'],
            'idToken'      => $response['idToken'],
            'refreshToken' => $response['refreshToken'],
            'firstName'    => $firstName,
        ];
    }

    /**
     * Sign in an existing user with email and password
     */
    public function signIn(string $email, string $password): array {
        $response = $this->request('signInWithPassword', [
            'email'             => $email,
            'password'          => $password,
            'returnSecureToken' => true,
        ]);

        if (isset($response['error'])) {
            return ['success' => false, 'error' => $this->parseError($response['error'])];
        }

        // Fetch user profile to get displayName
        $profile = $this->getUserProfile($response['idToken']);
        $firstName = $profile['displayName'] ?? '';

        return [
            'success'      => true,
            'uid'          => $response['localId'],
            'email'        => $response['email'],
            'idToken'      => $response['idToken'],
            'refreshToken' => $response['refreshToken'],
            'firstName'    => $firstName,
        ];
    }

    /**
     * Send a password reset email
     */
    public function sendPasswordReset(string $email): array {
        $response = $this->request('sendOobCode', [
            'requestType' => 'PASSWORD_RESET',
            'email'       => $email,
        ]);

        if (isset($response['error'])) {
            return ['success' => false, 'error' => $this->parseError($response['error'])];
        }

        return ['success' => true, 'email' => $response['email'] ?? $email];
    }

    /**
     * Get user profile data using their ID token
     */
    public function getUserProfile(string $idToken): array {
        $response = $this->request('lookup', [
            'idToken' => $idToken,
        ]);

        if (isset($response['error'])) {
            return [];
        }

        return $response['users'][0] ?? [];
    }

    /**
     * Refresh an expired ID token using a refresh token
     */
    public function refreshToken(string $refreshToken): array {
        $url = 'https://securetoken.googleapis.com/v1/token?key=' . $this->apiKey;

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_HTTPHEADER     => ['Content-Type: application/x-www-form-urlencoded'],
            CURLOPT_POSTFIELDS     => http_build_query([
                'grant_type'    => 'refresh_token',
                'refresh_token' => $refreshToken,
            ]),
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_TIMEOUT        => 30,
        ]);

        $body = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $data = json_decode($body, true);

        if ($httpCode !== 200 || isset($data['error'])) {
            return ['success' => false, 'error' => 'Session expired. Please sign in again.'];
        }

        return [
            'success'      => true,
            'idToken'      => $data['id_token'],
            'refreshToken' => $data['refresh_token'],
            'uid'          => $data['user_id'],
        ];
    }

    /**
     * Verify an ID token is still valid
     */
    public function verifyToken(string $idToken): bool {
        $profile = $this->getUserProfile($idToken);
        return !empty($profile);
    }

    /**
     * Make a request to the Firebase Identity Toolkit REST API
     */
    private function request(string $endpoint, array $payload): array {
        $url = self::BASE_URL . ':' . $endpoint . '?key=' . $this->apiKey;

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_HTTPHEADER     => ['Content-Type: application/json'],
            CURLOPT_POSTFIELDS     => json_encode($payload),
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_TIMEOUT        => 30,
        ]);

        $body = curl_exec($ch);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            return ['error' => ['message' => 'Connection error: ' . $error]];
        }

        return json_decode($body, true) ?: ['error' => ['message' => 'Invalid response from Firebase']];
    }

    /**
     * Parse Firebase error codes into user-friendly messages
     */
    private function parseError(array $error): string {
        $code = $error['message'] ?? 'UNKNOWN_ERROR';

        $messages = [
            'EMAIL_EXISTS'             => 'An account with this email already exists.',
            'EMAIL_NOT_FOUND'          => 'No account found with this email address.',
            'INVALID_PASSWORD'         => 'Incorrect password. Please try again.',
            'INVALID_LOGIN_CREDENTIALS'=> 'Invalid email or password. Please try again.',
            'USER_DISABLED'            => 'This account has been disabled.',
            'TOO_MANY_ATTEMPTS_TRY_LATER' => 'Too many failed attempts. Please try again later.',
            'WEAK_PASSWORD'            => 'Password must be at least 6 characters long.',
            'INVALID_EMAIL'            => 'Please enter a valid email address.',
            'OPERATION_NOT_ALLOWED'    => 'Email/password sign-in is not enabled in Firebase.',
            'MISSING_PASSWORD'         => 'Please enter a password.',
            'MISSING_EMAIL'            => 'Please enter an email address.',
        ];

        // Firebase sometimes appends extra info after a colon
        $baseCode = explode(' : ', $code)[0];

        return $messages[$baseCode] ?? 'Authentication error: ' . $code;
    }
}
